#!/bin/bash

# mountmedia.sh
#
# Usage: mountmedia.sh <mount point> <temp file> <log file> <read-only option> \
#                      <options> <option1> <userid>
#
# mount point: the mount point for the media 
#
# Return Codes:
# 0 - Media mounted successfully, and in read-write mode
# 1 - General failure, cause unknown
# 2 - Media already mounted, attempt to unmount failed
# 3 - Media mounted successfully, is write protected, which is not necessarily a problem
# 4 - Media device failure
# 5 - No media in the device
# 6 - Filesystem problem, probably resulting from unformatted media
# 7 - Attempted to explicitly mount read-only media (or perhaps on a read-only device) as read-write
#
# Author: C. Schuck
#
# 01 05/28/2002  P. Provost - Added chown -R command after mounting media

MOUNTPOINT=$1
TEMPFILE=$2
LOGFILE=$3
RO=$4
OPTIONS=$5
OPTION1=$6
USERID=$7

echo "-> mountmedia.sh" >> $LOGFILE
echo "Mounting the media at $MOUNTPOINT for `date`" >> $LOGFILE

actzTrace "XMDA   T: -> mountmedia.sh"

# Check to see if media is already mounted
actzTrace "XMDA   F: mountmedia.sh: checking /etc/mtab for mounted media at $MOUNTPOINT"
echo "mountmedia.sh: contents of /etc/mtab = " >> $LOGFILE
echo "" >> $LOGFILE
cat /etc/mtab >> $LOGFILE
echo "" >> $LOGFILE 
if grep -q $MOUNTPOINT /etc/mtab; then # Media is already mounted
    echo "mountmedia.sh: Media is already mounted at $MOUNTPOINT" >> $LOGFILE
    echo "mountmedia.sh: Unmounting media at $MOUNTPOINT" >> $LOGFILE
    actzTrace "XMDA   F: mountmedia.sh: Media is already mounted at $MOUNTPOINT" 
    actzTrace "XMDA   F: mountmedia.sh: Umounting media at $MOUNTPOINT" 
    if ! umount -v $MOUNTPOINT >> $LOGFILE 2>&1; then
        actzTrace "XMDA   F: mountmedia.sh: Couldn't unmount the media at $MOUNTPOINT...exiting"
       	echo "mountmedia.sh: Couldn't unmount the media.. exiting" >> $LOGFILE
       	exit 2
    else
        actzTrace "XMDA   F: mountmedia.sh: Unmounted the media at $MOUNTPOINT"
        echo "mountmedia.sh: Unmounted the media at $MOUNTPOINT" >> $LOGFILE
    fi
fi

# Mount
actzTrace "XMDA   F: mountmedia.sh: Mounting the media at $MOUNTPOINT"
actzTrace "XMDA   F: mountmedia.sh: mount -v$RO $MOUNTPOINT $OPTIONS $OPTION1" 
if mount -v$RO $MOUNTPOINT $OPTIONS $OPTION1 > $TEMPFILE 2>&1; then
    actzTrace "XMDA   F: mountmedia.sh: Mounted the media at $MOUNTPOINT"   
    echo "mountmedia.sh: Mounted the media" >> $LOGFILE
    if grep -q -i "write-protected" $TEMPFILE; then
        actzTrace "XMDA   F: mountmedia.sh: The media is write-protected, which may not be a problem.. setting appropriate return code" 
       	echo "mountmedia.sh: The media is write-protected, which may not be a problem.. setting appropriate return code" >> $LOGFILE
        ls -lR $MOUNTPOINT >> $LOGFILE
        actzTrace "XMDA   F: <- mountmedia.sh"
        echo "<- mountmedia.sh" >> $LOGFILE        
       	exit 3
    else
        actzTrace "XMDA   F: mountmedia.sh: The media is not write-protected."
        # Check to see if the userid was passed to the script
        if [ -z $USERID ]
        then
            # Userid was not passed, don't try to change the onwer of the filesystem
            echo "mountmedia.sh: No userid passed. Did not change the owner of the media" >> $LOGFILE
            ls -lR $MOUNTPOINT >> $LOGFILE
            if ! chmod -R 777 $MOUNTPOINT >> $LOGFILE 2>&1; then
                actzTrace "XMDA   F: mountmedia.sh: Did not change the permissions on the media"
                echo "mountmedia.sh: Did not change the permissions of the filesystem on the media" >> $LOGFILE
            else
                actzTrace "XMDA   F: mountmedia.sh: Changed the permissions on the media"
                echo "mountmedia.sh: Changed the permissions on the filesystem on the media" >> $LOGFILE
            fi
            ls -lR $MOUNTPOINT >> $LOGFILE
        else
            ls -lR $MOUNTPOINT >> $LOGFILE
            # Userid was passed, try to change the owner of the filesystem 
            if ! chown -R $USERID $MOUNTPOINT >> $LOGFILE 2>&1; then
                actzTrace "XMDA   F: mountmedia.sh: Did not change the owner of the media at $MOUNTPOINT"        
                echo "mountmedia.sh: Userid was passed but did not change the owner of the media" >> $LOGFILE
            else
                actzTrace "XMDA   F: mountmedia.sh: Changed the owner of the media at $MOUNTPOINT" 
                echo "mountmedia.sh: Userid was passed and changed the owner of the media to $USERID" >> $LOGFILE
            fi
            ls -lR $MOUNTPOINT >> $LOGFILE
            if ! chmod -R 777 $MOUNTPOINT >> $LOGFILE 2>&1; then
                actzTrace "XMDA   F: mountmedia.sh: Did not change the permissions on the media"
                echo "mountmedia.sh: Did not change the permissions of the filesystem on the media" >> $LOGFILE
            else
                actzTrace "XMDA   F: mountmedia.sh: Changed the permissions on the media"
                echo "mountmedia.sh: Changed the permissions on the filesystem on the media" >> $LOGFILE
            fi
            ls -lR $MOUNTPOINT >> $LOGFILE 
        fi        
    fi
else
    actzTrace "XMDA   F: mountmedia.sh: Couldn't mount the media at $MOUNTPOINT"
    echo "Couldn't mount the media.." >> $LOGFILE
    if grep -q -i "wrong major or minor number" $TEMPFILE; then
        actzTrace "XMDA   F: mountmedia.sh: Hardware failure on the media drive...exiting" 
       	echo "Hardware failure on the media drive.. exiting" >> $LOGFILE
        exit 4
    elif (grep -q -i "No medium found" $TEMPFILE) || (grep -q -i "is not a valid block device" $TEMPFILE); then
        actzTrace "XMDA   F: mountmedia.sh: No media in the device...exiting" 
       	echo "No media in the device.. exiting" >> $LOGFILE
        exit 5
    elif (grep -q -i "wrong fs type" $TEMPFILE) || (grep -q -i "you must specify the filesystem type" $TEMPFILE); then
        actzTrace "XMDA   F: mountmedia.sh: Wrong filesystem type, or unformatted media, or wrong media type...exiting"
       	echo "Wrong filesystem type, or unformatted media, or wrong media type.. exiting" >> $LOGFILE
        exit 6
    elif grep -q -i "is write-protected but explicit" $TEMPFILE; then
        actzTrace "XMDA   F: mountmedia.sh: Attempted to mount read-only media as read-write...exiting" 
       	echo "Attempted to mount read-only media as read-write.. exiting" >> $LOGFILE
        exit 7
    else
        actzTrace "XMDA   F: mountmedia.sh: Unknown error...exiting" 
       	echo "Unknown error.. exiting" >> $LOGFILE
    	exit 1
    fi
fi

actzTrace "XMDA   T: <- mountmedia.sh"

echo "<- mountmedia.sh" >> $LOGFILE
echo "" >> $LOGFILE

exit 0

